install.packages("tidyverse") #run if you have never installed tidyverse before. ggplot2 is included in the tidyverse package, as well as some other cool packages like dplyr and tidyr.

library(tidyverse) #run at the start of each new session.


# Data -------------------------------------------------------------

#use R built-in data set called "iris"
iris <- iris %>% 
  dplyr::select(Species,Sepal.Length,Sepal.Width,Petal.Length,Petal.Width) %>%
  as_tibble()

#use gather(), separate() and spread() - from tidyr - to create iris.wide

#gather: key = column name representing new variable, -column names not to gather, value = column name representing new variable values. Gather collects a set of column names and places into “key” column, collects cells under those columns and places into “value”.

#separate: col= column to separate, into = (names of new columns), sep = where to split the variables.

iris.wide <- iris %>% mutate(Flower = 1:nrow(iris)) %>% #add column with unique IDs
  gather(key="Key", value="Value", -Species, -Flower) %>%
  separate(col=Key, into=c("Part", "Measure"), sep="\\.") %>%
  spread(key=Measure,value=Value) %>%
  dplyr::select(-Flower)

#use gather() and separate() - from tidyr - to create iris.tidy
iris.tidy <- iris %>% gather(key="Key", value="Value", -Species) %>%
  separate(col=Key, into=c("Part", "Measure"), sep="\\.") 


#examples of graphs using the three above dataframes
ggplot(iris, aes(x=Sepal.Length, y=Sepal.Width)) +
  geom_point()

ggplot(iris.wide, aes(x=Length, y=Width)) +
  geom_point()

ggplot(iris.tidy, aes(x=Measure, y=Value)) +
  geom_point(position=position_jitter(width=0.2))



# Aesthetics --------------------------------------------------------------

#Attribute of geom_point is set to "red"
ggplot(iris.wide, aes(x=Length, y=Width)) +
  geom_point(col = "red")

#Aesthetic mapping is set to colour by Species
ggplot(iris.wide, aes(x=Length, y=Width, col=Part)) +
  geom_point()

#Manually change the colour of the aesthetic mapping by Species (using "scale_colour_manual()"). Change the axis labels and legend labels.
ggplot(iris.wide, aes(x=Length, y=Width, col=Part)) +
  geom_point() +
  scale_colour_manual(values=c("orchid3","orange1"), labels=c("Iris Petal", "Iris Sepal"), name="Floral Part") +
  xlab("Length (cm)") +
  ylab("Width (cm)")



# Geometries and Statistics -----------------------------------------------

#Create means and standard deviations for variables from iris dataframe
iris.summary <- iris %>% 
  group_by(Species) %>%
  summarise_all(funs(mean,sd))

#An example of setting the main aesthetic inside the ggplot() - displays points from the iris dataframe, and calling other data sets inside specific geom aesthetics - to get the mean & sd from iris.summary dataframe.
ggplot(iris, aes(x=Petal.Length, y=Petal.Width, col=Species)) +
  geom_point() +
  geom_point(data=iris.summary, aes(x=Petal.Length_mean, y=Petal.Width_mean), size=7, shape=21, fill="grey60") +
  geom_errorbar(data=iris.summary, aes(x=Petal.Length_mean, y=Petal.Width_mean, ymin=Petal.Width_mean-Petal.Width_sd, ymax=Petal.Width_mean+Petal.Width_sd), width=0.2) +
  geom_errorbarh(data=iris.summary, aes(x=Petal.Length_mean, y=Petal.Width_mean, xmin=Petal.Length_mean-Petal.Length_sd, xmax=Petal.Length_mean+Petal.Length_sd), height=0.1)


#geom_point, geom_smooth and stat_smooth examples
ggplot(iris, aes(x=Sepal.Length, y=Sepal.Width, col=Species)) +
  geom_point() +
  geom_smooth() #use “span” to control wiggliness, se=FALSE to remove standard error ribbon

iris.lm <- ggplot(iris, aes(x=Sepal.Length, y=Sepal.Width, col=Species)) +
  geom_point() +
  stat_smooth(method="lm") #also loess, glm, gam, MASS::rlm

ggplot(iris, aes(x=Sepal.Length, y=Sepal.Width, col=Species)) +
  geom_point() +
  stat_smooth(method="lm", fullrange=TRUE) #predict beyond data


#geom_bar vs stat_summary
ggplot(iris, aes(x=Species, y=Petal.Length, fill=Species)) +
  geom_bar(stat="summary", fun.y=mean, width=0.5) +
  geom_errorbar(stat="summary", fun.data=mean_sdl, width=0.1)

  ggplot(iris, aes(x=Species, y=Petal.Length, fill=Species)) +
  stat_summary(geom="bar", fun.y=mean, width=0.5) +
  stat_summary(geom="errorbar", fun.data=mean_sdl, width=0.1) 
  
  
#geom_boxplot instead of geom_bar
#to position TukeyHSD letters, try using the max value of each factor and nudge a little upwards
  Tukey <- iris %>% group_by(Species) %>% 
  summarise(value = max(Petal.Length)) %>%
  mutate(letters = c("a", "b", "c"))

ggplot(iris, aes(x=Species, y=Petal.Length, fill=Species)) +
  geom_boxplot(width=0.5) +
  geom_text(data=Tukey, aes(x=Species, y=value, label=letters), nudge_y=0.5) +
  scale_x_discrete(labels=c("I.setosa","I.versicolor","I.virginica")) 

  
#geom_histogram
ggplot(iris, aes(x=Sepal.Width)) +
  geom_histogram() #default binwidth = range/30

ggplot(iris, aes(x=Sepal.Width)) +
  geom_histogram(binwidth=0.1) #choose own binwidth

ggplot(iris, aes(x = Sepal.Width)) +
  geom_histogram(aes(y=..density..), binwidth = 0.1) #plot density instead of counts


#geom_line with geom_ribbon. Also change legend title with "labs".
shoot.root <- read_csv("Shoot_Root.csv")

ggplot(shoot.root, aes(x=Day, y=Shoot_Root, col=factor(Treatment), fill=factor(Treatment))) +
  geom_point(stat="summary", fun.y=mean, size=3) +
  geom_line(stat="summary", fun.y=mean) +
  geom_ribbon(stat="summary", fun.data=mean_cl_boot, alpha=0.3, col=NA) +
  labs(colour="Treatment") +
  labs(fill="Treatment")


# Facets ------------------------------------------------------------------

ggplot(iris.wide, aes(x=Length, y=Width, col=Species)) +
  geom_point() +
  facet_grid(.~Part) #row ~ column. Use fullstop for non-defined section.


# Coordinates -------------------------------------------------------------

iris.lm

iris.lm + scale_x_continuous(limits = c(4.5, 5.5)) + theme(aspect.ratio = 1) #removes data

iris.lm + coord_cartesian(xlim = c(4.5, 5.5)) + theme(aspect.ratio = 1) #really just zooms in

# Themes ------------------------------------------------------------------

#use R built-in data set called "mtcars"
mtcars <- mtcars

cars <- ggplot(mtcars, aes(x=factor(cyl), y=mpg)) +
  geom_boxplot() +
  theme_grey() #default

cars + theme_bw()

cars + theme_classic()


#This is the definition of theme_bw() straight from ggplot2's source code:
theme_bw <- function(base_size = 12) {
  structure(list(
    axis.line =         theme_blank(),
    axis.text.x =       theme_text(size = base_size * 0.8 , lineheight = 0.9, vjust = 1),
    axis.text.y =       theme_text(size = base_size * 0.8, lineheight = 0.9, hjust = 1),
    axis.ticks =        theme_segment(colour = "black", size = 0.2),
    axis.title.x =      theme_text(size = base_size, vjust = 1),
    axis.title.y =      theme_text(size = base_size, angle = 90, vjust = 0.5),
    axis.ticks.length = unit(0.3, "lines"),
    axis.ticks.margin = unit(0.5, "lines"),
    
    legend.background = theme_rect(colour=NA), 
    legend.key =        theme_rect(colour = "grey80"),
    legend.key.size =   unit(1.2, "lines"),
    legend.text =       theme_text(size = base_size * 0.8),
    legend.title =      theme_text(size = base_size * 0.8, face = "bold", hjust = 0),
    legend.position =   "right",
    
    panel.background =  theme_rect(fill = "white", colour = NA), 
    panel.border =      theme_rect(fill = NA, colour="grey50"), 
    panel.grid.major =  theme_line(colour = "grey90", size = 0.2),
    panel.grid.minor =  theme_line(colour = "grey98", size = 0.5),
    panel.margin =      unit(0.25, "lines"),
    
    strip.background =  theme_rect(fill = "grey80", colour = "grey50"), 
    strip.text.x =      theme_text(size = base_size * 0.8),
    strip.text.y =      theme_text(size = base_size * 0.8, angle = -90),
    
    plot.background =   theme_rect(colour = NA),
    plot.title =        theme_text(size = base_size * 1.2),
    plot.margin =       unit(c(1, 1, 0.5, 0.5), "lines")
  ), class = "options")
}

#How I would do things, an example:

ggplot(iris.wide, aes(x=Length, y=Width, col=Species)) +
  geom_jitter(size = 3, alpha = 0.5) +
  facet_grid(.~Part) +
  theme_bw() +
  theme(panel.grid = element_blank(),
        axis.line = element_blank(),
        text = element_text(size = 16),
        legend.title = element_blank(),
        legend.position = "bottom",
        legend.key.size = unit(1.5, "lines"),
        legend.text = element_text(face = "italic", size = 14),
        strip.background = element_rect(fill = "grey90", colour = NA),
        axis.title.y = element_text(hjust=0.57, margin = margin(r = 15)))



